/**************************************************************\
 *  Prototypes
\**************************************************************/
void codec_out(short x);
short codec_in(void);
void codec_init(void);
void init_HWI(void);
void emif_init(void);


/**************************************************************\
 *  Include files
\**************************************************************/
#include <c6x.h>           // C6000 compiler definitions

#include <csl.h>           // CSL headers
#include <csl_irq.h>
#include <csl_mcbsp.h>

#include <bsl.h>           // BSL headers
#include <bsl_ad535.h>     // -i build option allows CCS to find these headers
#include <bsl_suggestions.h>

/**************************************************************\
 *  Global Variables
\**************************************************************/
    /**********************************************************\
     *  The following AD535 Handle and Config structure were   
     *  entered verbatim from the BSL documentation. The config
     
     0
     *  values are common to many applications; if nothing 
     *  else, they make a good starting point.               
    \**********************************************************/
    AD535_Handle hAD535;             //gestisce il canale di codec                          
    AD535_Config my_AD535_Config = {     //configura il canale di codec                      
        AD535_LOOPBACK_DISABLE,                                
        AD535_MICGAIN_OFF,                                     
        AD535_GAIN_0DB,                                        
        AD535_GAIN_0DB                                         
    };                                                         

/**************************************************************\
 *  Main Routine
\**************************************************************/
void main()
{
    CSL_init();              // Initialize CSL library
    BSL_init();              // Initialize BSL library
    emif_init();             // Dummy function - see note below
    codec_init();
    init_HWI();
 
    AD535_read(hAD535);  /* Write anything to cause mcbsp
                                to start transmit interrupts */

    /**************************************************\
     *  Rather than creating an infinite loop in main,
     *  DSP/BIOS provides an infinite idle (IDL) loop.
     *  All you need to do is return from main. We'll
	 *  use IDL in Lab 5 and 6 by returning from main()
     *  to DSP/BIOS's IDL function.
    \**************************************************/                  
    while(1);
}

/**************************************************************\
 *	Configure External Memory Interface (EMIF)
 *
 *  We have a dummy function here to setup the EMIF. In a "real"
 *  system you would be required to setup the EMIF but in our
 *  development system CCS sets up the memory for us (refer to 
 *  the DSK6211_6711.gel file for the CCS script that 
 *  accomplishes this task.
\**************************************************************/
void emif_init(void){
}

/*************************************************************\
 *	Initialize hardware interrupts. 
 **************************************************************
 *  The AD535 codec is hardwired to McBSP0. Therefore,
 *	our system uses the McBSP0 receive interrupt 
 *  (RINT0) to signal when to write the next output sample.
 *	
 *	Enabling the CPU to recognize the RINT0 interrupt requires:
 *
 *	1. Map McBSP0 receive interrupt (RINT0) to interrupt 11 (HWI11)
 *     using the Config Tool
 *	2. Enable interrupts globally
 *	3. Enable McBSP transmit interrupt
\*************************************************************/
void init_HWI(void)
{
	IRQ_globalEnable();             // Enable ints globally
	IRQ_enable(IRQ_EVT_RINT0);		// Enable McBSP0 interrupt
}

//Interrupt service routine you need to reference in HWI module of CDB file
void RINT0_HWI(void)
{

	short a,e,c,d;
	a=codec_in()*4; //prendiamo il segnale in ingresso
	e=fir1(a);      //Filtro antialiasing(passa banda)
	c=(sineGen()*e)/65536; //Modulazione con moltiplicazione in modo da avere nei
	                        //32 bit la parte pi significativa
	d=fir(c);       //Filtro finale  (passa basso)                                     
	codec_out(d);   //Portiamo in uscita il nostro segnale
	
	
	                   
}

/**********************************************************\
 *	Codec Functions
\**********************************************************/
void codec_init()
{
	/* Use the BSL routines to:  Open, reset, & config the AD535               *\
	 * Part of the AD535_open() routine opens and configures the McBSP0, since *
	\*  all communications with the AD535 codec are handled via the McBSP0.    */
	hAD535 = AD535_open(AD535_localId);      //   Open AD535 and set hAD535 handle
	AD535_reset(hAD535);                     //   Reset AD535                     
	AD535_config(hAD535, &my_AD535_Config);  //   Configure AD535                 
	
	/**************************************************\
	 *	This sets a bit in a McBSP register thus
	 *	allowing it to continue running even when the
	 *	debugger is halted. This keeps the codec from
	 *	being corrupted.
	 *
	 *  We've put this function in the "bsl_suggestions"
	\**************************************************/
	MCBSP_setfree(0);
}

/******************************************************\
 *	Output a sine sample from the function SineGen()
 *	to the AD535 codec.
\******************************************************/
void codec_out(short x) 
{ 
	AD535_HWI_write(hAD535, x); //scrive i dati da spedire
}

short codec_in() 
{ 
	return AD535_HWI_read(hAD535); //legge il dato ricevuto
}







